
const Sequelize = require("sequelize");
const {models} = require("../models");
const {paginate} = require("../helpers/paginate");

let currentPage;

exports.load = async (req, res, next, postId) => {

    try {

        const post = await models.Post.findByPk(postId, {
            include: [
                {model: models.Attachment, as: 'attachment'},
                {model: models.User, as: 'user'}
            ]
        });
        if (post) {
            req.load = {...req.load, post};
            next();
        } else {
            throw new Error('No hay ningún post con el id=', postId);
        }

    } catch (err) {
        next(err);
    }

}

exports.index = async (req, res, next) => {

    try {

        const count = await models.Post.count();

        const items_per_page = 3;

        // The page to show is given in the query
        const pageno = parseInt(req.query.pageno) || 1;
        currentPage = pageno;

        // Create a String with the HTMl used to render the pagination buttons.
        // This String is added to a local variable of the res object.
        res.locals.paginate_control = paginate(count, items_per_page, pageno, req);

        const findOptions = {
            include: [
                {model: models.Attachment, as: 'attachment'},
                {model: models.User, as: 'user'}
            ],
            offset: items_per_page * (pageno - 1),
            limit: items_per_page
        };

        const posts = await models.Post.findAll(findOptions);
        res.render('posts/index.ejs', {posts});
        
    } catch (error) {
        next(error);
    }
}

exports.show = async(req, res, next) => {

    const {post} = req.load;

    // var md = require('markdown-it')({
    //     html:         false,        // Enable HTML tags in source
    //     xhtmlOut:     true,        // Use '/' to close single tags (<br />).
    //     breaks:       true,        // Convert '\n' in paragraphs into <br>
    //     linkify:      false,        // Autoconvert URL-like text to links
    //     typographer:  false
    //   });

    post.body = post.body.replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/\n/g, "<br/>");  // "<" y ">" antes que "\n" para que no me cambie los "<br/>"
    // post.body = md.render(post.body);

    res.render('posts/show', {post, currentPage});
}

exports.new = async (req, res, next) => {

    if (!req.session.loginUser) {
        res.redirect('/login');
        return;
    }

    const post = {
        title: "",
        body: ""
    };

    res.render('posts/new', {post});
}

exports.create = async (req, res, next) => {

    const {title, body} = req.body;

    const userId = req.session.loginUser.id;

    try {
        if(title == '' || body == ''){
            req.session.message = {
              message: 'Please insert the requested information.'
            }
            res.redirect('/posts/new')
        }
        let post = models.Post.build({
            title,
            body,
            userId
        });

        post = await post.save({fields: ["title", "body", "userId"]});
        console.log('Success: Post created successfully.');

        try {
            if (!req.file) {
                console.log('Info: Post without attachment.');
                return;
            }

            // Create the post attachment
            await createPostAttachment(req, post);

        } catch (error) {
            console.log('Error: Failed to create attachment: ' + error.message);
        } finally {
            res.redirect('/posts/' + post.id);
        }
    } catch (error) {
        if (error instanceof (Sequelize.ValidationError)) {
            console.log('There are errors in the form:');
            error.errors.forEach(({message}) => console.log(message));
            let post = models.Post.build({
                title,
                body
            });
            res.render('posts/new', {post});
        } else {
            next(error);
        }
    }
}

const createPostAttachment = async (req, post) => {

    const img = req.file.buffer.toString('base64');
    const url = `${req.protocol}://${req.get('host')}/posts/${post.id}/attachment`;

    // Create the new attachment into the data base.
    const attachment = await models.Attachment.create({
        mime: req.file.mimetype,
        img,
        url
    });
    await post.setAttachment(attachment);
    console.log('Success: Attachment saved successfully.');
};

exports.edit = async (req, res, next) => {

    if (!req.session.loginUser) {
        res.redirect('/login');
        return;
    }
    
    const {post} = req.load;
    
    console.log(post.id, req.session.loginUser.id);
    if(post.userId != req.session.loginUser.id){
        //Avisa de alguna manera
        res.redirect(`/posts/?pageno=${currentPage}`);
        return;
    }


    res.render('posts/edit', {post});
}

exports.update = async (req, res, next) => {

    const {post} = req.load;

    post.title = req.body.title;
    post.body = req.body.body;

    try {
        if(post.title == '' || post.body == ''){
            req.session.message = {
              message: 'Please insert the requested information.'
            }
            res.redirect(`/posts/${post.id}/edit`)
        }
        await post.save({fields: ["title", "body"]});
        console.log('Success: Post edited successfully.');

        try {
            if (!req.file) {
                console.log('Info: Post attachment not changed.');
                return;
            }

            //Delete old attachment.
            if (post.attachment) {
                await post.attachment.destroy();
                await post.setAttachment();
            }
            
            // Create the post attachment
            await createPostAttachment(req, post);

        } catch (error) {
            console.log('Error: Failed saving the new attachment: ' + error.message);
        } finally {
            res.redirect('/posts/' + post.id);
        }
    } catch (error) {
        if (error instanceof (Sequelize.ValidationError)) {
            console.log('There are errors in the form:');
            error.errors.forEach(({message}) => console.log(message));
            res.render('posts/edit', {post});
        } else {
            next(error);
        }
    }
}

exports.destroy = async (req, res, next) => {

    if (!req.session.loginUser) {
        res.redirect('/login');
        return;
    }

    if (req.session.loginUser.id != req.load.post.userId || req.session.loginUser.id != 1) {
        res.redirect('/posts');
        return;
    }

    const attachment = req.load.post.attachment;

    try {
        await req.load.post.destroy();
        attachment && await attachment.destroy();
        console.log('Success: Post deleted successfully.');
        res.redirect('/posts');
    } catch (error) {
        console.log('Error deleting the Post: ' + error.message);

        next(error);
    }
}

exports.attachment = (req, res, next) => {

    const {post} = req.load;

    const {attachment} = post;

    if (!attachment) {
        res.redirect("/images/none.png");
    } else if (attachment.img) {
        res.type(attachment.mime);
        res.send(Buffer.from(attachment.img.toString(), 'base64'));
    } else if (attachment.url) {
        res.redirect(attachment.url);
    } else {
        res.redirect("/images/none.png");
    }
}
