const { where } = require("sequelize");
const Sequelize = require("sequelize");
const {models} = require("../models");
const post = require("../models/post");

// Autoload the user with id equals to :userId
exports.load = async (req, res, next, userId) => {

    try {

        const user = await models.User.findByPk(userId, {
            include: [
                {model: models.ProfileImg, as: 'profileImg'},
                {model: models.Post, as:'post'}
            ]
        });
        if (user) {
            req.load = {...req.load, user};
            next();
        } else {
            throw new Error('Error: There is no user with id=' + userId);
        }
    } catch (error) {
        next(error);
    }
};


// GET /users
exports.index = async (req, res, next) => {

    try {
        const findOptions = {
            order: ['username']
        };

        const users = await models.User.findAll(findOptions);
        res.render('users/index', {users});
    } catch (error) {
        next(error);
    }
};

// GET /users/:userId
exports.show = async (req, res, next) => {

    const {user} = req.load;

    const findOptions = {
        include: [
            {model: models.User, as: 'user'}
        ],
        where: {
            userId: user.id
        }
    };

    const posts = await models.Post.findAll(findOptions);

    res.render('users/show', {user, posts});
};


// GET /users/new
exports.new = async (req, res, next) => {

    const user = {
        username: "",
        email: "",
        password: ""
    };

    res.render('users/new', {user});
};


// POST /users
exports.create = async (req, res, next) => {

    const {username, email, password} = req.body;

    let user = models.User.build({
        username,
        email,
        password
    });

    // Password must not be empty.
    if (!password) {
        console.log('Error: Password must not be empty.');
        return res.render('users/new', {user});
    }

    try {
        // Save into the data base
        user = await user.save({fields: ["username", "email", "password", "salt"]});
        console.log('Success: User created successfully.');

        try {
            if (!req.file) {
                console.log('Info: User without profile picture.');
                return;
            }
            // Create the User profile image
            await createUserProfileImg(req, user);
        } catch (err) {
            console.log('Error: Failed to create profile picture: ' + err.message);
        } finally {
            if (req.session.loginUser) {
                res.redirect('/users/' + user.id);
            } else {
                res.redirect('/login'); // Redirection to the login page
            }
        }
    } catch (error) {
        if (error instanceof Sequelize.UniqueConstraintError) {
            console.log(`Error: User "${username}" already exists.`);
            res.render('users/new', {user});
        } else if (error instanceof Sequelize.ValidationError) {
            console.log('Error: There are errors in the form:');
            error.errors.forEach(({message}) => console.log('Error:', message));
            res.render('users/new', {user});
        } else {
            next(error);
        }
    }
};

const createUserProfileImg = async (req, user) => {

    const img = req.file.buffer.toString('base64');
    const url = `${req.protocol}://${req.get('host')}/users/${user.id}/profileImg`;

    // Create the new attachment into the data base.
    const profileImg = await models.ProfileImg.create({
        mime: req.file.mimetype,
        img,
        url
    });
    await user.setProfileImg(profileImg);
    console.log('Success: Attachment saved successfully.');
};

// GET /users/:userId/edit
exports.edit = (req, res, next) => {

    const {user} = req.load;

    res.render('users/edit', {user});
};


// PUT /users/:userId
exports.update = async (req, res, next) => {

    const {body} = req;
    const {user} = req.load;

    // user.username  = body.user.username; // edition not allowed

    let fields_to_update = [];

    // ¿Cambio el password?
    if (body.password) {
        console.log('Updating password');
        user.password = body.password;
        fields_to_update.push('salt');
        fields_to_update.push('password');
    }

    try {
        await user.save({fields: fields_to_update});
        console.log('Success: User updated successfully.');
        try {
            if (!req.file) {
                console.log('Info: User profile picture not changed.');
                return;
            }
            if (user.profileImg) {
                await user.profileImg.destroy();
                await user.setProfileImg();
            }
            // Create the post attachment
            await createUserProfileImg(req, user);
        } catch (err) {
            console.log('Error: Failed saving the new profile picture: ' + err.message);
        }
    } catch (error) {
        if (error instanceof Sequelize.ValidationError) {
            console.log('Error: There are errors in the form:');
            error.errors.forEach(({message}) => console.log('Error:', message));
            res.render('users/edit', {user});
        } else {
            next(error);
        }
    } finally {
        res.redirect('/users/' + user.id);
    }
};


// DELETE /users/:userId
exports.destroy = async (req, res, next) => {

    try {
        // Deleting logged user.
        if (req.session.loginUser?.id === req.load.user.id) {
            // Close the user session
            delete req.session.loginUser;
        }

        const profileImg = req.load.user.profileImg;
        await req.load.user.destroy()
        profileImg && await profileImg.destroy();
        console.log('Success: User deleted successfully.');
        res.redirect('/users');
    } catch (error) {
        console.log('Error deleting the User:', error.message);
        next(error);
    }
};

exports.profileImg = (req, res, next) => {
    const {user} = req.load;
    const {profileImg} = user;

    if (!profileImg) {
        res.redirect('/images/none.png');
    } else if (profileImg.img) {
        res.type(profileImg.mime);
        res.send(Buffer.from(profileImg.img.toString(), 'base64'));
    } else if (profileImg.url) {
        res.redirect(profileImg.url);
    } else {
        res.redirect("/images/none.png");
    }
}